// content.js (Firefox / WebExtensions)
browser.runtime.onMessage.addListener((message) => {
  if (!message || message.action !== "getTables") return;

  const extractTableName = (tableEl) => {
    const clean = (s) => (s || "").replace(/\s+/g, " ").trim();

    const caption = tableEl.querySelector("caption");
    if (caption && clean(caption.innerText)) return clean(caption.innerText);

    const ariaLabel = tableEl.getAttribute("aria-label");
    if (ariaLabel && clean(ariaLabel)) return clean(ariaLabel);

    const card = tableEl.closest(
      "[data-ui-name='Card'], [data-test-batch-table], section, article, [role='region']"
    );
    if (card) {
      const semrushTitle = card.querySelector(
        "[data-test-batch-table-title] h1, [data-test-batch-table-title] h2, [data-test-batch-table-title] h3"
      );
      if (semrushTitle && clean(semrushTitle.innerText)) return clean(semrushTitle.innerText);

      const heading = card.querySelector("h1, h2, h3, h4, h5, h6");
      if (heading && clean(heading.innerText)) return clean(heading.innerText);
    }

    let el = tableEl;
    for (let i = 0; i < 6 && el; i++) {
      let prev = el.previousElementSibling;
      while (prev) {
        const h =
          prev.querySelector?.("h1,h2,h3,h4,h5,h6") ||
          (/^H[1-6]$/.test(prev.tagName) ? prev : null);

        if (h && clean(h.innerText)) return clean(h.innerText);
        prev = prev.previousElementSibling;
      }
      el = el.parentElement;
    }

    return null;
  };

  const getTablesNow = () => {
    const tables = [];

    const htmlTables = Array.from(document.querySelectorAll("table"));
    if (htmlTables.length) {
      htmlTables.forEach((table) => {
        const rows = Array.from(table.rows).map((row) =>
          Array.from(row.cells).map((cell) => cell.innerText.trim())
        );

        if (rows.length) {
          tables.push({ rows, name: extractTableName(table) });
        }
      });
      return tables;
    }

    const rowDivs = Array.from(document.querySelectorAll("div[role='row']"));
    if (rowDivs.length) {
      const rows = rowDivs.map((row) =>
        Array.from(row.querySelectorAll("div[role='gridcell']")).map((cell) =>
          cell.innerText.trim()
        )
      );

      if (rows.length) tables.push({ rows, name: null });
    }

    return tables;
  };

  return new Promise((resolve) => {
    let tries = 0;
    const maxTries = 12;

    const timer = setInterval(() => {
      const tables = getTablesNow();

      if (tables.length || tries >= maxTries) {
        clearInterval(timer);
        resolve({ tables });
      }

      tries++;
    }, 250);
  });
});
